// Nom du snack : eni_react_native_villes_suppression
// Nom du script : App.js
// Auteur : Christian VIGOUROUX
// Date de création : 11/07/2024
// Date de dernière modification : 11/07/2024 (Christian VIGOUROUX)
// Objet : Affichage d'une liste de villes
//         avec une lecture depuis une table MySQL
//         et possibilité de suppression
// URL : https://snack.expo.dev/@cvigouroux/eni_react_native_villes_suppression

// Import des composants React Native et React
import React, { useState, useEffect } from 'react';
import { SafeAreaView, Text, StyleSheet, View, FlatList, Button, Alert } from 'react-native';

// Fonction principale App
export default function App() {

  // Constantes de la fonction App
  const [dataSource, setDataSource] = useState([]);

  // Méthode useEffect
  // NB : Accès aux données de l'application distante (format JSON)
  //      URL : http://christian-vigouroux.fr/react/php_mysql_liste_complete_villes.php
  useEffect(() => {
    listeVilles();
  }, []);

  // Fonction listeVilles (liste des villes)
  const listeVilles = () => {
    fetch('http://christian-vigouroux.fr/react/php_mysql_liste_complete_villes.php')
      .then((response) => response.json())
      .then((responseJson) => {
        setDataSource(responseJson);
      })
      .catch((error) => {
        console.error('Erreur dans la recherche des villes :', error);
      });
  };

  // Fonction suppressionVille (suppression de la ville sélectionnée)
  const suppressionVille = (id) => {

    // Appel du script PHP de suppression (méthode GET)
    // avec passage de l'ID de la ville en paramètre
    fetch(`http://christian-vigouroux.fr/react/php_mysql_suppression_ville.php?id_ville=${id}`, {
      method: 'GET',
    })
    .then((response) => {
      console.log('Response status:', response.status);
      return response.json();
    })
    .then((responseJson) => {
      console.log('Réponse JSON:', responseJson);
      if (responseJson.success) {
        Alert.alert('Succès', 'Ville supprimée avec succès');
        listeVilles();
      } else {
        Alert.alert('Erreur', 'Une erreur est survenue lors de la suppression de la ville : ' + responseJson.message);
      }
    })
    .catch((error) => {
      Alert.alert('Erreur', 'Une erreur est survenue lors de la suppression de la ville');
      console.error('Erreur de suppression de la ville :', error);
    });
    
  };

  // Affichage de l'interface utilisateur
  return (

    // Vue de type SafeAreaView 
    <SafeAreaView style={{ flex: 1 }}>
      
      {/* Vue d'affichage de la Flat List */}
      <View style={styles.container}>
        
        {/* Composant FlatList */}
        <FlatList
          
          // Récupération des données
          data={dataSource}
          keyExtractor={(item) => item.id_ville.toString()}
          
          // Affichage de l'item
          renderItem={({ item }) => (
            <View style={styles.itemContainer}>
        
              {/* Affichage des champs id_ville et nom_ville  */}
              <Text style={styles.itemText}>
                {item.id_ville} - {item.nom_ville}
              </Text>
        
              {/* Affichage du bouton de suppression */}
              <Button
                title="Suppression"
               onPress={() => suppressionVille(item.id_ville)}
              />

            </View>
          )}
          
          // Affichage d'un trait de séparation
          ItemSeparatorComponent={ItemSeparatorView}
        
        />

      </View>

    </SafeAreaView>

  );

}

// Fonction ItemSeparatorView (affichage d'un trait de séparation)  
const ItemSeparatorView = () => {
  return (
    <View
      style={{
        height: 5,
        width: '100%',
        backgroundColor: '#C8C8C8',
      }}
    />
  );
};

// Style CSS
const styles = StyleSheet.create({
  container: {
    backgroundColor: 'white',
    flex: 1,
  },
  itemContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 10,
  },
  itemText: {
    fontSize: 16,
  },
  button: {
    backgroundColor: 'red',
    color: 'white',
    padding: 10,
  },
});